/**
* \file: change_cmd.c
*
* \version: $Id:$
*
* \release: $Name:$
*
* \component: authorization level daemon
*
* \author: Rexaline Xavier  /  RexalineInfancia.Xavier@in.bosch.com
*
* \copyright (c) 2017 Advanced Driver Information Technology.
* This code is developed by Advanced Driver Information Technology.
* Copyright of Advanced Driver Information Technology, Bosch, and DENSO.
* All rights reserved.
*
*
***********************************************************************/

#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include <unistd.h>
#include <errno.h>

#include "common.h"

#include "util/helper.h"
#include "util/logger.h"
#include "util/cmdline_parser.h"
#include "util/cfgfile_parser.h"

static cfg_item_t levelnum_cfg 		= {.spec = &levelnum_cfg_spec, .cmdline = &levelnum_cmdline_spec};
static cfg_item_t levelprivkey_cfg 	= {.spec = &levelprivkey_cfg_spec, .cmdline = &levelprivkey_cmdline_spec};
static cfg_item_t serialnum_cfg 	= {.spec = &serialnum_cfg_spec, .cmdline = &serialnum_cmdline_spec};
static cfg_item_t ecuid_cfg 		= {.spec = &ecuid_cfg_spec, .cmdline = &ecuid_cmdline_spec};
static cfg_item_t help_cfg			= {.spec = &help_cfg_spec, .cmdline = &help_cmdline_spec };

static error_code_t change_cmd_init(void);
static error_code_t change_cmd_parse_args(const char *binary_name, int argc, char *argv[]);
static error_code_t change_cmd_start(void);
static void change_cmd_deinit(void);

static cfg_item_t *const change_cmd_cfgs[] = {
		&levelnum_cfg,
		&levelprivkey_cfg,
		&serialnum_cfg,
		&ecuid_cfg,
		&help_cfg
};

command_vtable_t change_cmd_vtable = {
        .command = "change",
        .command_description = "Changes the system security level of ALD to given level",
        .init = change_cmd_init,
        .parse_args = change_cmd_parse_args,
        .start = change_cmd_start,
        .deinit = change_cmd_deinit
};

static void change_cmd_print_usage(const char *binary_name)
{
	printf("\nUsage: %s %s [options...]\n\n",binary_name, change_cmd_vtable.command);
	printf("\t"LEVELNUM_CMD_SHORT", "LEVELNUM_CMD_LONG"\t\tLevel number to which the ALD shall change\n");
	printf("\t"LEVELPRIVKEY_CMD_SHORT", "LEVELPRIVKEY_CMD_LONG"\tPath to file containing the key to unlock the requested level\n");
	printf("\t"SERIALNUM_CMD_SHORT", "SERIALNUM_CMD_LONG"\t\tID of device or person requesting the level change\n");
	printf("\t"ECUID_CMD_SHORT", "ECUID_CMD_LONG"\t\tID defining the type or version of the device\n");
	printf("\t"HELP_CMD_SHORT", "HELP_CMD_LONG"\t\tDisplays this help and exits\n\n");
}

static void change_cmd_print_help(const char *binary_name)
{
	printf("\nAuthorization Level Daemon Change Level Utility\n"
			"\t- Used to change the system security level with the help of the ADIT authorization level daemon\n\n");
	printf("Usage: %s %s [options...]\n\n",binary_name, change_cmd_vtable.command);
	printf("\t"LEVELNUM_CMD_SHORT", "LEVELNUM_CMD_LONG"\t\tLevel number to which ALD changes level\n");
	printf("\t"LEVELPRIVKEY_CMD_SHORT", "LEVELPRIVKEY_CMD_LONG"\tPrivate key path needed to sign level change request\n");
	printf("\t"SERIALNUM_CMD_SHORT", "SERIALNUM_CMD_LONG"\t\tTextual id for token used for authorization\n");
	printf("\t"ECUID_CMD_SHORT", "ECUID_CMD_LONG"\t\tDefines the type of hardware device that is used\n");
	printf("\t"HELP_CMD_SHORT", "HELP_CMD_LONG"\t\tDisplays this help and exits\n\n");
	printf("The level to which ALD has to change can be set with '"LEVELNUM_CMD_LONG"' option.\n"
			"If it is not passed, the ALD level will be changed to default level of '%d'.\n"
			"The path of the private key used to sign the challenge/request data can be set\n"
			"with '"LEVELPRIVKEY_CMD_LONG"' option. If the path is not passed the private key\n"
			"is taken from default path : '%s'.\n"
			"The textual identifier for the token used for authorization(e.g. smartcard,USB stick)\n"
			"or the person that did the authorization can be set with option '"SERIALNUM_CMD_LONG"'.\n"
			"The serial number is a textual ID with the maximum of %u characters. Exceeding characters are stripped.\n"
			"If this is not passed the default value of '%s' is used.\n"
			"The option '"ECUID_CMD_LONG"' sets the expected type or version of hardware device (e.g. XYZ02 HW Version 3).\n"
			"The ecuid is a textual ID with a maximum size of %u characters, Exceeding characters are stripped.\n"
			"If this is not passed,the default value of '%s' is used.\n\n", helper_get_U32_default(levelnum_cfg.spec),
			helper_get_str_default(levelprivkey_cfg.spec),
			RESPONSE_SERIAL_NUMBER_SIZE, helper_get_str_default(serialnum_cfg.spec),
			CHALLENGE_ECU_ID_SIZE, helper_get_str_default(ecuid_cfg.spec));
}

static error_code_t change_cmd_init(void)
{
	logger_init_console_only(LOGGER_LEVEL_ERROR);

	return common_init();
}

static error_code_t change_cmd_parse_args(const char *binary_name, int argc, char *argv[])
{
	error_code_t result = RESULT_OK;
	int change_cmd_cfgs_cnt = ARRAY_SIZE(change_cmd_cfgs);

	helper_items_init(change_cmd_cfgs, change_cmd_cfgs_cnt);

	result = cmdline_parser_parse_cmd_line(change_cmd_cfgs, change_cmd_cfgs_cnt, argv, argc);

	if (result == RESULT_INVALID_ARGS)
	{
		change_cmd_print_usage(binary_name);
		return RESULT_INVALID_ARGS;
	}

	if (helper_get_flag(&help_cfg))
	{
		change_cmd_print_help(binary_name);
		return RESULT_HELP_PRINTED;
	}
	return result;
}

static error_code_t change_cmd_start(void)
{
	Change_level *change_level_iface;
	error_code_t result = RESULT_OK;
	challenge_response_t response;

	result = common_connect_ALD(&change_level_iface);

    common_initialize_response(&response,
                               (security_level_t)helper_get_U32_value(&levelnum_cfg),
                               helper_get_str_value(&serialnum_cfg));

	if (result == RESULT_OK)
	{
		printf("--------------------- Requesting a challenge ... ---------------------------------------------\n");
		result = common_get_challenge_and_add_to_response(change_level_iface,&response,helper_get_str_value(&ecuid_cfg));
	}

	if (result == RESULT_OK)
	{
		printf("--------------------- calculating a signature ... --------------------------------------------\n");
		result = common_calculate_signature(helper_get_str_value(&levelprivkey_cfg),&response);
	}

	if (result == RESULT_OK)
	{
		printf("--------------------- sending the response ... -----------------------------------------------\n");
		result = common_send_response(change_level_iface,&response);
	}

	common_disconnect_ald(&change_level_iface);

	if (result == RESULT_OK)
		printf("Level changed successfully to %d.\n",(int)response.targeted_level);

	return result;
}

static void change_cmd_deinit(void)
{
	common_deinit();
}
